/*
 * BatchExport.jsx
 * Javascript for InDesign CS5
 * Version date: 20120425
 *
 * PURPOSE:
 * ========
 * Automated folder walker to batch-create exports from InDesign documents.
 * Walks a folder structure, opens any InDesign document and calls the 
 * main export script 'ExportCurrentDocument.jsx'.
 *
 * Instructions:
 * -------------
 * Make a copy of this Script into the folder where 'ExportCurrentDocument.jsx' is located (one up).
 * then create an export settings file:
 * Call 'ExportCurrentDocument.jsx' from InDesign's Scripts palette and make your settings.
 * Cancel the dialog. This creates the file 'settings.set' in the sub folder 'zz_Settings' within your home folder: ~BatchXSLT4InDesignComm/Scripts/.
 * Rename this 'settings.set' to identify the purpose of these settings like 'settings_Product1.set'.
 *
 * Make a copy of the file 'BatchExportExample.jt' within the folder 'zz_Settings' and name it 'BatchExport.jt'.
 * With a plain text editor edit the job tickets within the lines
 * # ************ JobTickets Section starts here
 * and 
 * # ************ JobTickets Section ends here
 * Two jobtickets are already contained but are disabled with the commands "state=0"
 * Settings descriptions may be found after the line # ************ Comments Section
 * You may edit, add, delete any of of the jobtickets to match your needs.
 *
 * Any number of jobtickets may be defined to act on subfolders within the base batch folder 
 * "~/Export/BATCHin"
 * The core connection between a jobticket and a certain folder is the keyword "jtSubFolder" witch 
 * must point to a folder within "~/Export/BATCHin".
 * Example:
 * a jobticket with the entry
 * jtSubFolder=myNews
 * will act on the folder path
 * ~/Export/BATCHin/myNews
 * 
 * Call the script from the InDesign Script palette and the export will start with variable settings for each batch folder.
 *
 *
 * IMPORTANT NOTES
 * ===============
 * 1. MAKE SURE, YOUR MACHINE (INDESIGN) HAS A LOT OF MEMORY AVAILABLE!
 *    Best is 4GB... or it may crash.
 * 2. INSTALL ALL FONTS USED IN THE DOCUMENTS!
 *    Otherwise the InDesign documents will re-flow with replaced fonts.
 * 3. InDesign document file names must end with '.indd'
 * 4. InDesign document file names and folder names MAY NOT CONTAIN SLASHES  '/' OR BACKSLASHES  '\'
 *
 * DISCLAIMER:
 * ===============
 * Absolutely no warranty. Use it as is or modify it to match your needs.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE PRODUCER OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Author: Andreas Imhof, www.aiedv.ch
 */

// ****** Configurable SETTINGS ******
var debugMode = 0;      // if set to 1, all messages also are written into the log file
var silentMode = "9";   // -1 = absolutely no message and dialog - just do it!
                        // 0 = show all dialogs and messages: main export dialog and processing and completion messages
                        // 1 = no main dialog
                        // 2 = no error message (silently quit)
                        // 4 = no processing messages
                        // 8 = no completion message
                        // Example: 9 = no main dialog and no completion message
var exporterScriptName = "ExportCurrentDocument.jsx";
                        // the main export script to call when a document is opened

var logFilepath = "~";  // default to user's home folder
var logFilename = "BatchXSLT_Batch_log.txt";	// set to "" to supress log file writing



var packageExtension = "V6";               // set to an extension like '_PE' to a dedicated package installation or empty for the standard package installation
                                           // this expands the main application installation path to
                                           // BXSLT4InDesignBasePath = "~/BatchXSLT4InDesign" + packageExtension + "/BatchXSLT/"
                                           // as well as the communications folder to
                                           // jobticketFilePath = "~/BatchXSLT4InDesignComm" + packageExtension + "/Scripts/zz_Settings"
var jobticketFileName = "BatchExport.jt";  // name of the JobTicket file to load. default = "BatchExport.jt"
var jobticketFilePath = "~/BatchXSLT4InDesignComm" + packageExtension + "/Scripts/zz_Settings";     // path to the JobTicket file. default in user home = "~/BatchXSLT4InDesignComm/Scripts/zz_Settings"
var settingsFilePathAlt = "zz_Settings";   // if it can not be found there try here: InDesignApplFolder/Scripts Panel/BatchXSLT4InDesign Scripts/zz_Settings

var loop = -1;                             // main loop delay before processing all jobtickets again in ms. set to -1 to run only once
                                           // also settable within jobticketFileName
var delayBetweenJobtickets = 2000;         // delay between jobtickets in ms. set to -1 to immediately run next jobticket
                                           // also settable within jobticketFileName

// ****** END OF Configurable SETTINGS ******

// jobticket variables
var jobticketFile = null;
var jobticket = new Object();					// the controls sent to ExportCurrentDocument.jsx
jobticket.state = "1";							// 0 = active jobticket, 0 = disabled
jobticket.description = "";						// the description text to a jobticket
jobticket.settingsFileName = "settings.set";	// the settings file to use;
jobticket.silentMode = "9";						// Flags to combine for desired Messages
												// -1 = absolutely no message and dialog - just do it!
												// 0 = show all dialogs and messages: main export dialog and processing and completion messages
												// 1 = no main dialog
												// 2 = no error message (silently quit)
												// 4 = no processing messages
												// 8 = no completion message
												// Example: 9 = no main dialog and no completion message
jobticket.mainExportFolder = "~/Export";		// the mainExportFolder for BatchXSLT read from settingsfile
jobticket.mainExportFolderIn = jobticket.mainExportFolder + "/in";		// the mainExportFolderIn for BatchXSLT
jobticket.mainExportFolderBatchInBase = jobticket.mainExportFolder + "/BATCHin";
jobticket.mainExportFolderBatchIn = jobticket.mainExportFolderBatchInBase;
												// The default input path for InDesign documents and images
												// (a path starting with tilde '~' points to the users home directory)
												// Usually, this should point to the 'BATCHin' folder of the main export folder structure
jobticket.jtSubFolder = "";						// the name of the subfolder below 'mainExportFolderBatchIn' this jboticket is valid for
jobticket.exportSubPath = "";					// the current subpath of a document
jobticket.BXSLT4InDesignBasePath = "~/BatchXSLT4InDesign" + packageExtension + "/BatchXSLT/";	// path to BXSLT4InDesign read from settingsfile
jobticket.BXSLTTriggerFile = "_BXSLT_trigger.go";	// starting BatchXSLT's transform process with a trigger file
													// set to "" to not to start BatchXSLT with a trigger file - you will start BatchXSLT for InDesign manually
jobticket.xslcss_relpath_override = "";	// fixed relative path for XSLCSS folder like "../../../XSLCSS/" (with ending slash) or empty to calculate
jobticket.noHTMLconversion = "0";		// 1 to NOT to create a HTML equivalent to XML file
jobticket.renameAfterDone = "_DONE";	// set to any string which is added to jtSubFolder after export is done like "_DONE"
										// leave empty to not to rename
jobticket.currentDocumentFilePathName = "";	// the path and name to the actually opened document file


// ERROR codes
function error() {
}
error.NOERR = 0;
error.WRONG_INDD_VERSION = -1;
error.EXPORTPATH_NOT_FOUND = -2;
error.OPENENIG_DOC_FAILURE = -3;
error.WRONG_SCRIPT_POSITION = -4;
error.JT_FILE_NOT_FOUND = -5;
error.NO_ACTIVE_JTS = -6;	// no jobtickets or all 'state' set to 0
error.JT_ERROR_SETTINGSFILE = -7;	// the defined settings file could not be read or found
error.JT_READY = 1;	// this is not an error - run this jobticket

function complete() {
}
complete.NOTHING_TODO = 0;
complete.EXPORTED_DOC = 1;

var sucessDocsCnt = 0;
var failedDocsCnt = 0;

var scriptName = "BatchExport";
// more vars
var overwriteExistingFiles = true;					// true to always recreate INX, pages PDF and JPEG
													// false to check if they exist. if YES continue with next
var activeScriptFullPath = app.activeScript;
var activeScriptName = getScriptName();

var appLanguage = 0;	// 0 = english
						// 1 = german
						// 2 = french
setLocale();

var inxExists = false;
var pdfExists = false;
var jpgExists = false;
var appVersion = "";
var os = "";
var cancelCommand = "";


/*********** Main loop START ***********/
app.scriptArgs.clear();	// safety clear all script arguments if this script was aborted
main();

app.scriptArgs.clear();	// safety clear all script arguments

/*********** Main loop END ***********/

// initialize and start export evtl. in a loop
function main() {
	// do some main init
	var initok = init();
	if (initok != error.NOERR) return;

	do {
		sucessDocsCnt = 0;
		failedDocsCnt = 0;
		var do_continue = false;
		var jt_ok = loadJobticket(jobticket);
		//alert("jt_ok: " + jt_ok);
		if (jt_ok == error.JT_READY) {	// we have read valid Jobticket
			message("****** JobTicket: "+ jobticket.description);
			writeLog("****** JobTicket START: "+ jobticket.description,true);
			if (false) {	// just to debug jobtickets read
				writeLog("      ****** JobTicket description: '" + jobticket.description + "'",true);
				writeLog("      settingsFileName: '" + jobticket.settingsFileName + "'",true);
				writeLog("      silentMode: '" + jobticket.silentMode + "'",true);
				writeLog("      STATUS: '" + jt_ok + "'",true);
			}
			var retval = initExport();
			app.scriptArgs.clear();
			do {
				if (retval == error.EXPORTPATH_NOT_FOUND) break;
				if (retval == complete.NOTHING_TODO) break;
	
				// show number of processed documents in this folder
				writeLog(loc(3,sucessDocsCnt,jobticket.mainExportFolderBatchIn),true);
				writeLog(loc(4,failedDocsCnt,jobticket.mainExportFolderBatchIn),true);
	
				// when ever a jobticket is done, kick off the transformer
				jobticket.currentDocumentFilePathName = "";
				//alert("jobticket.mainExportFolder: "+ jobticket.mainExportFolder + "\njobticket.mainExportFolderIn: "+ jobticket.mainExportFolderIn + "\njobticket.jtSubFolder: "+ jobticket.jtSubFolder + "\njobticket.exportSubPath: "+ jobticket.exportSubPath);
				//start_exporterScript(jobticket.jtSubFolder, 2, true);	// start BatchXSLT now without doing a document export
				start_exporterScript("", 2, true);	// start BatchXSLT now without doing a document export
				// get return values
				var ECD_exitcode = 0;	// true = succsessfully started, otherwise failure
				try { if ( app.scriptArgs.isDefined("ECD_exitcode") ) success = app.scriptArgs.getValue("ECD_exitcode");
				} catch(e) {}
				if (ECD_exitcode == error.NOERR) writeLog(loc(24),true);
				else writeLog(loc(25),true);
	
				// evtl, rename Batchin folder
				renameBatchInFolder();
				writeLog("****** JobTicket END: "+ jobticket.description,true);
			} while(false);
			if (delayBetweenJobtickets > 0) {
				message(loc(5,delayBetweenJobtickets,cancelCommand));	// waiting...
				$.sleep(delayBetweenJobtickets);
				messageClose();
			}
			continue;
		}
		else {	// handle errors
			switch (jt_ok) {
				case error.JT_FILE_NOT_FOUND: return; 	// error message already shown
				case error.NOERR: break; 	// no more JobTickets to process
				case error.NO_ACTIVE_JTS: {
					message(loc(8,jobticketFileName, jobticketFilePath));	// no active jobtickets - quit
					$.sleep(10000); messageClose();
					return;
					}
				case error.JT_ERROR_SETTINGSFILE: {	// settings file define in JT not found
					writeLog(loc(27,jobticket.settingsFileName,jobticket.description),true);
					message(loc(27,jobticket.settingsFileName,jobticket.description));
					$.sleep(10000); messageClose();
					do_continue = true;	// try to continue with other JTs
					break;
					}
			}
		}
		//alert("do_continue:" + do_continue);
		if (do_continue) continue;
		if (loop < 0) break;
		message(loc(6,loop,cancelCommand));	// waiting...
		$.sleep(loop);
		messageClose();
	} while (true);

	return;
}


/****************************
 * init some stuff.....
 */
function init() {
	app.scriptArgs.clear();
	var scriptpath = getScriptsPath();
	if (scriptpath.toLowerCase().indexOf("zz_utilities") >= 0) {	// script at wrong folder
		alert("Please, read the usage instructions within this script!\nMove this script one folder up and call it again!\n\nVerschieben Sie dieses Script um einen Ordner nach oben und rufen Sie es erneut auf!\n\n");
		return(error.WRONG_SCRIPT_POSITION);
	}

	os = $.os;
	do {
		if (os.toLowerCase().indexOf("macintosh") >= 0) { 
			os = "mac";
			cancelCommand = "[Apple][.]";
			break;
		}
		if (os.toLowerCase().indexOf("wind") >= 0) {
			os = "win";
			cancelCommand = "[Ctrl][c]";
			break;
		}
		os = "unix";
		cancelCommand = "[Ctrl][c]";
	} while(false);

	// get application language
	switch (app.locale) {
		case Locale.germanLocale:	// 1279477613
			appLanguage = 1;	// talk german
			break;
		case Locale.frenchLocale:	// 1279477362
			appLanguage = 2;	// talk french
			break;
		default:
			appLanguage = 0;	// talk english
	}
	//appLanguage = 1;	// set to 0,1,2 to force a language for messages

	// run with CS2 versions 4.x.x. , CS3 5.x.x and CS4 6.x.x

	appVersion = app.version;
	var INDmajorversion = parseInt(appVersion.charAt(0));
	if (INDmajorversion >= 5) is_CS3_or_later = true;
	else is_CS3_or_later = false;

	if (INDmajorversion < 7) {	// at least CS5
		//alert("This export script may not run with this version of InDesign!...");
		alert(loc(0));
		return(error.WRONG_INDD_VERSION);
	}
	return(error.NOERR);
}


function initExport() {

	// check if Export folder structure may be found
	if (jobticket.jtSubFolder != "") {	// in batch mode add the source main folder
		if ((endsWith(jobticket.mainExportFolderBatchIn,"/") == false) && (startsWith(jobticket.jtSubFolder,"/") == false)) jobticket.mainExportFolderBatchIn += "/";
		jobticket.mainExportFolderBatchIn += jobticket.jtSubFolder;
	}
	if (jobticket.mainExportFolderBatchIn != "") {
		var f = new File(jobticket.mainExportFolderBatchIn);
		if (f.exists == false) {
			message(loc(1,jobticket.mainExportFolderBatchIn,"(code " + error.EXPORTPATH_NOT_FOUND +")"));	// Export folder not found at:\n%%1%%\n\ncontinuing scanning...
			// we recall our self or the next jobticket
			$.sleep(3000); messageClose();
			return(error.EXPORTPATH_NOT_FOUND);
		}
	}


	// start processing documents
	var had2work = initFolderWalker(jobticket.mainExportFolderBatchIn);
	if (had2work == true) {
	}
	else {
		message(loc(2,jobticket.mainExportFolderBatchIn));	// Nothing has been exported from folder \n\n%%1%%
		$.sleep(1500); messageClose();
	}

	return( ((had2work == true) ? complete.EXPORTED_DOC : complete.NOTHING_TODO) );
}


/****************************
 * here we go.....
 */
function initFolderWalker(exportFolder) {

	if ( (exportFolder == null) || (exportFolder == "") ) return(false);
	var baseExportFolder = new Folder(exportFolder);

	// change some settings
	var scrpref = app.scriptPreferences;
	scrpref.userInteractionLevel = UserInteractionLevels.neverInteract;
	var old_display_settings = app.displayPerformancePreferences.defaultDisplaySettings
	app.displayPerformancePreferences.defaultDisplaySettings = ViewDisplaySettings.highQuality;

	if (false) {	// debugging the settings (set to false for production)
		var message = "Export Settings:";
		message += "\nbaseExportFolder.fsName: " + baseExportFolder.fsName;
		alert(message);
		return(true);
	}

	// now walk the folders
	var had2export = walkFolder(baseExportFolder,0);	// call the folder walker and exporter, return true if we exported something, otherwise false

	// restore settings
	app.displayPerformancePreferences.defaultDisplaySettings = old_display_settings;
	scrpref.userInteractionLevel = UserInteractionLevels.interactWithAll;
	$.gc();

	// return false if we had nothing to do
	if (had2export == false) return(false);
	return(true);
}



/****************************
 * walk a folder tree and export all .indd files
 */
var had2work = false;
function walkFolder(theFolder,call_level) {	// walk a folder object
	if (theFolder == null) return(false);
	if (call_level == 0) had2work = false;
	var contentArr = theFolder.getFiles();	// get all contained files and folders

	for (var i = 0; i < contentArr.length; i++) {
		if (contentArr[i] instanceof Folder) 	{	// is a folder: recall us
			var fldrhad2work = false;
			if (endsWith(contentArr[i].name.toLowerCase(),jobticket.renameAfterDone.toLowerCase()) == false) 
				fldrhad2work = walkFolder(contentArr[i],++call_level);
			if (had2work == false) had2work = fldrhad2work;	// dont set if already set to true
		}
		else {	// is a file
			if (contentArr[i].name.indexOf(".") == 0) continue;	// is hidden Unix file
			var is_indd_file = false;	// true when this is indesign document
			var is_qxd_file = false;	// true when this is Quark document
			do {
				// check if it is a .indd file
				if (endsWith(contentArr[i].name.toLowerCase(),".indd") == true) { is_indd_file = true; break; }
				if (endsWith(contentArr[i].name.toLowerCase(),".qxd") == true) { is_qxd_file = true; break; }
				if (contentArr[i].name.indexOf(".") < 0) {	// has no filename extension - might be one on OSX
					var ftype = detectFileType(contentArr[i]);
					if (ftype == "indd") { is_indd_file = true; break; }
					is_qxd_file = isXPressFile(contentArr[i]);
				}
			} while (false);
			if (is_indd_file || is_qxd_file) {	// call exporter script
				var path = getPath(contentArr[i].fsName,contentArr[i].name);
				//alert("walkFolder():\ncontentArr[i]: " + contentArr[i] + "\n\npath:" + path + "\n\nname: " + contentArr[i].name + "\n\nfullName: " + contentArr[i].fullName + "\n\nabsoluteURI: " + contentArr[i].absoluteURI + "\n\nfsName: " + contentArr[i].fsName);
				var exported = callExporter(contentArr[i], path);
				if (exported != error.NOERR) {
				}
				had2work = true;
				$.gc();
				$.sleep(200);
			}
		}
	}
	return(had2work);
}

/****************************
 * Open a document, call pages export and close it
 */
function callExporter(theFileToExport, theExportPath) {
	// open INDD or QXP document
	var myDocument = null;
	var myDocumentWin = null;

	var saveRedrawSetting = true;
	try {
		var saveRedrawSetting = app.scriptPreferences.enableRedraw;
		app.scriptPreferences.enableRedraw = true;
	} catch (Exception) {} 


	$.gc();
	//alert("callExporter()\ntheFileToExport: " + theFileToExport.fsName + "\n\ntheExportPath: " + theExportPath);
	try {
		myDocument = app.open(theFileToExport,false);	// do not show window now! this prevents some dialogs from beeing shown!
		jobticket.currentDocumentFilePathName = theFileToExport.fsName;
	}
	catch (Exception) {} 
	finally {
		if (myDocument == null) { 
			writeLog(loc(26,theFileToExport.name,theExportPath),true);
			failedDocsCnt++; return(error.OPENENIG_DOC_FAILURE);
		}
		myDocumentWin = myDocument.windows.add();		// show the document window ...
		//var bounds = new Array(100,100,900,750);		// ... and size it
		//myDocumentWin.bounds = bounds;
		myDocumentWin.bringToFront();
		$.gc();
		$.sleep(1000);	// let window breath
	}


	// write log entry
	writeLog("--------------------------------",true)
	writeLog(loc(20,theFileToExport.name,theExportPath,jobticket.settingsFileName),true)

	// call the exporter script 'ExportCurrentDocument.jsx'
	var experr = start_exporterScript(theExportPath, 1, null);	// but don't start BatchXSLT
	// get return values
	var ECD_exitcode = error.NOERR;	// 0 = ok, otherwise failure
	try { if ( app.scriptArgs.isDefined("ECD_exitcode") ) success = app.scriptArgs.getValue("ECD_exitcode");
	} catch(e) {}

	if (ECD_exitcode == error.NOERR) {
		sucessDocsCnt++;
	}
	else {
		failedDocsCnt++;	// error
		writeLog(loc(23,ECD_exitcode,theFileToExport.name,theExportPath),true);
	}

	// write 'finished' 
	writeLog(loc(21,ECD_exitcode));
	// close it
	try {
		app.activeDocument.close();
	} catch(e) {}
	writeLog(loc(22,theFileToExport.name));
	try {
		app.scriptPreferences.enableRedraw = saveRedrawSetting;
	} catch (Exception) {} 
	$.gc();
	$.sleep(500);
	return(ECD_exitcode);
}


function detectFileType(thefile) {
	if (thefile.exists == false) return ("");
	var chars = "";
	var numchars = 10;
	try {
		thefile.encoding = "BINARY";
		var err = thefile.open("r");
		chars = thefile.read(numchars);
	}
	catch (Exception) {} 
	finally {
		thefile.close();
	}
	if (chars.length < numchars) return("");
/*	for (var i=0; i < 8; i++) {
		alert("filename: " + thefile.name + "\nchars: " + chars + "\ni: " + i + " " + chars.charCodeAt(i));
	}*/
	// InDesign files start with hex: 06 06   ED  F5  D8 1D 46  E5
	//							 dec:  6  6  237 245 216 29 70 229
	if (   (chars.charCodeAt(0) == 6)
		&& (chars.charCodeAt(1) == 6)
		&& (chars.charCodeAt(2) == 237)
		&& (chars.charCodeAt(3) == 245)
		&& (chars.charCodeAt(4) == 216)
		&& (chars.charCodeAt(5) == 29)
		&& (chars.charCodeAt(6) == 70)
		&& (chars.charCodeAt(7) == 229)
		) { return ("indd"); }

	return ("");
}


/************************
 * call the exporter script 'ExportCurrentDocument.jsx'
 */
function start_exporterScript(theExportPath, triggerMode, finalPath) {
	//alert("start_exporterScript theExportPath: " + theExportPath);
	if (finalPath == null || finalPath == false) jobticket.exportSubPath = theExportPath.substr(jobticket.mainExportFolderBatchIn.length+1);
	else jobticket.exportSubPath = theExportPath;
	//writeLog("jobticket.mainExportFolder:'" + jobticket.mainExportFolder + "'",true);
	//writeLog("jobticket.exportSubPath:'" + jobticket.exportSubPath + "'",true);
	// call our main export script with parameters
	//alert("BatchExport: start_exporterScript()" + "\ntheExportPath: " + theExportPath + "\nfinalPath: " + finalPath + "\njobticket.mainExportFolder: " + jobticket.mainExportFolder + "\njobticket.mainExportFolderBatchIn: "+ jobticket.mainExportFolderBatchIn + "\njobticket.jtSubFolder: "+ jobticket.jtSubFolder + "\njobticket.exportSubPath: "+ jobticket.exportSubPath);

	app.scriptArgs.setValue("callerScript", scriptName);
	app.scriptArgs.setValue("exportSubPath", jobticket.exportSubPath);
	app.scriptArgs.setValue("settingsFileName", jobticket.settingsFileName);
	app.scriptArgs.setValue("mainExportFolder", jobticket.mainExportFolder);
	app.scriptArgs.setValue("suppress_exportSubPath", ""+finalPath);
	app.scriptArgs.setValue("jtSubFolder", jobticket.jtSubFolder);
	app.scriptArgs.setValue("xslcss_relpath_override", jobticket.xslcss_relpath_override);
	var tm = "" + triggerMode;
	app.scriptArgs.setValue("BatchXSLTtriggerMode", tm);	// 1 = don't start BatchXSLT! we will do that when jobticket is done
																	// 2 = start BatchXSLT only (no export)
	app.scriptArgs.setValue("noHTMLconversion", jobticket.noHTMLconversion);
	app.scriptArgs.setValue("silentMode", "" + jobticket.silentMode);
	app.scriptArgs.setValue("currentDocumentFilePathName", "" + jobticket.currentDocumentFilePathName);
	var scriptFile = new File(getScriptsPath() + "/" + exporterScriptName);
	try {
		app.doScript(scriptFile,ScriptLanguage.javascript);
	}
	catch (Exception) {
		writeLog("Unexpected Exeption calling script: " + scriptFile, true);
	}
}


/************************
 * load jobticket settings from File
 */
var num_active_jobtickets = 0;
function loadJobticket(jobticket) {
	// check if file must be opened or if we can continue to read tickets
	if (jobticketFile == null) {
		num_active_jobtickets = 0;	// reset count
		var jtpath = "";
		if (jobticketFilePath.indexOf("/") >= 0) jtpath = jobticketFilePath;	// take full path if given
		else jtpath = getScriptsPath() + "/" + jobticketFilePath;				// otherwise try to get from InDesigns Script Panel folder
		jobticketFile = new File(jtpath + "/" + jobticketFileName);
		//alert("jobticketFile pathname:\n" + jobticketFile.fsName);
		if (jobticketFile.exists == false) {
			copySettingsFiles();
		}
		if (jobticketFile.exists == false) {
			jobticketFile = null;
			writeLog(loc(7,jobticketFileName,jtpath),true);
			message(loc(7,jobticketFileName,jtpath));	// jt file not found - quit
			$.sleep(10000);
			messageClose();
			return(error.JT_FILE_NOT_FOUND);
		}
		var err = jobticketFile.open("r");
	}

	var line = "";
	var endOfJobTickets = false;
	var settingsFileLoaded = -1;	// init status of setting file to error
	while (!endOfJobTickets && !jobticketFile.eof) {
		line = jobticketFile.readln();
		if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
			var key = line.substr(0,line.indexOf("="));
			var arg = line.substr(line.indexOf("=")+1);
			if (line.indexOf("=") >= 0) {
				key = line.substr(0,line.indexOf("="));
				arg = line.substr(line.indexOf("=")+1);
			}
			else {
				key = line;
				arg = "";
			}
			//alert(line + "\n----->\n" + key+"="+arg);
			var args_arr = new Array(key,arg);
			switch(args_arr[0]) {
				case "loop": loop = parseInt(args_arr[1]); break;
				case "delayBetweenJobtickets": delayBetweenJobtickets = parseInt(args_arr[1]); break;
				case "description": jobticket.description = args_arr[1]; break;
				case "settingsFileName":
					jobticket.settingsFileName = args_arr[1];
					settingsFileLoaded = loadSettingsFromFile(jobticket.settingsFileName);	// get some export values from this settings file
					break;
				case "mainExportFolder": jobticket.mainExportFolder = getFsName(args_arr[1]); break;
				case "mainExportFolderBatchIn": jobticket.mainExportFolderBatchIn = getFsName(args_arr[1]); break;
				case "jtSubFolder": jobticket.jtSubFolder = args_arr[1]; break;
				case "BXSLT4InDesignBasePath": jobticket.BXSLT4InDesignBasePath = args_arr[1]; break;
				case "BXSLTTriggerFile": jobticket.BXSLTTriggerFile = args_arr[1]; break;
				case "xslcss_relpath_override": jobticket.xslcss_relpath_override = args_arr[1]; break;
				case "silentMode": jobticket.silentMode = args_arr[1]; break;
				case "noHTMLconversion": jobticket.noHTMLconversion = args_arr[1]; break;
				case "renameAfterDone": jobticket.renameAfterDone = args_arr[1]; break;
				case "go":
					num_active_jobtickets++;
					if (settingsFileLoaded != error.NOERR) return(error.JT_ERROR_SETTINGSFILE);
					return(error.JT_READY);
					break;

				case "exit":	// fall through
				case "endOfJobTickets": endOfJobTickets = true;
					break;

				case "state": jobticket.state = args_arr[1];
					if (jobticket.state == "0") {		// skip to end of this jobticket
						jobticket.state = "1";
						while (!jobticketFile.eof) {
							line = jobticketFile.readln();
							//writeLog("++++skip line: '" + line + "'",true);	// just to debug jobtickets read
							if ((line != null) && (line != "")) {
								if (line == "go") break;
							}
						}
					}
					break;
			}
		}
	}
	if ((endOfJobTickets == true) || jobticketFile.eof) {
		err = jobticketFile.close();
		jobticketFile = null;
	}
	if (num_active_jobtickets <= 0) return(error.NO_ACTIVE_JTS);
	return(error.NOERR);
}


/****************************
 * load export settings from .set file
 */
function loadSettingsFromFile(theSettingsFileName) {
	// load the requested settings from disk
	var jtpath = "";
	if (jobticketFilePath.indexOf("/") >= 0) jtpath = jobticketFilePath;	// take full path if given
	else jtpath = getScriptsPath() + "/" + jobticketFilePath;				// otherwise try to get from InDesigns Script Panel folder
	var settingsFile = new File(jtpath + "/" + theSettingsFileName);
	//alert("settingsFile pathname:\n" + settingsFile.fsName);
	if (settingsFile.exists == false) return(-1);
	var err = settingsFile.open("r");

	var line = "";
	var found_settings = 0;
	while (!settingsFile.eof && (found_settings < 2)) {	// currently we need 2 settings only
		line = settingsFile.readln();
		if ((line != null) && (line != "") && (line.indexOf("#") != 0)) {
			var key = line.substr(0,line.indexOf("="));
			var arg = line.substr(line.indexOf("=")+1);
			var args_arr = new Array(key,arg);
			switch(args_arr[0]) {
				case "BXSLT4InDesignBasePath": jobticket.BXSLT4InDesignBasePath = args_arr[1]; found_settings++; break;
				/* this is currently not used here
				case "mainExportFolder": jobticket.mainExportFolder = args_arr[1]; found_settings++; break;
				case "companyName": companyName = args_arr[1]; break;
				case "objectName": objectName = args_arr[1]; break;
				case "objectShortcut": objectShortcut = args_arr[1]; break;
				case "issueYear": issueYear = args_arr[1]; break;
				case "issueDate": issueDate = args_arr[1]; break;
				case "cssName": cssName = args_arr[1]; break;
				case "magnifyingGlass": magnifyingGlass = parseInt(args_arr[1]); break;
				case "outputMode": outputMode = parseInt(args_arr[1]); break;
				case "exportImages": if (args_arr[1] == "true") exportImages = true; else exportImages = false; break;
				case "exportPageJPEGs": if (args_arr[1] == "true") exportPageJPEGs = true; else exportPageJPEGs = false; break;
				case "exportPagePDFs": if (args_arr[1] == "true") exportPagePDFs = true; else exportPagePDFs = false; break;
				case "exportDocumentPDF": if (args_arr[1] == "true") exportDocumentPDF = true; else exportDocumentPDF = false; break;
				case "imageScale": imageParameters = args_arr[1]; break;	// backward compatibility
				case "imageParameters": imageParameters = args_arr[1]; break;
				case "pageImageWidth": pageImageWidth = args_arr[1]; break;
				case "splittingStoriesOption": splittingStoriesOption = args_arr[1]; break;
				case "chainedBoxesNoCatch": if (args_arr[1] == "true") chainedBoxesNoCatch = true; else chainedBoxesNoCatch = false; break;
				case "copyOriginalImageFiles": if (args_arr[1] == "true") copyOriginalImageFiles = true; else copyOriginalImageFiles = false; break;
				case "TABLE_BORDER_COLLAPSE": TABLE_BORDER_COLLAPSE = args_arr[1]; break;
				case "TABLE_CELLS_WIDTH_PRESERVE": TABLE_CELLS_WIDTH_PRESERVE = args_arr[1]; break;
				case "closeDocumentAfterDone": if (args_arr[1] == "true") closeDocumentAfterDone = true; else closeDocumentAfterDone = false; break;
				case "allowRevertDocument": if (args_arr[1] == "true") allowRevertDocument = true; else allowRevertDocument = false; break;
				case "BatchXSLTtriggerMode": BatchXSLTtriggerMode = parseInt(args_arr[1]); break;
				case "showExportFolder": if (args_arr[1] == "true") showExportFolder = true; else showExportFolder = false; break;
				case "saveSettings": if (args_arr[1] == "true") saveSettings = true; else saveSettings = false; break;
				case "showMessage": if (args_arr[1] == "true") showMessage = true; else showMessage = false; break;
				*/
			}
		}
	}
	err = settingsFile.close();
	return(0);
}


function renameBatchInFolder() {
	if (jobticket.renameAfterDone == "") return(0);
	var batchfldr = new Folder(jobticket.mainExportFolderBatchIn);
	var ok = 0;
	if (jobticket.jtSubFolder != "") {
		ok = batchfldr.rename(jobticket.jtSubFolder + jobticket.renameAfterDone);
	}
	else {	// get all sub folders and rename them
		var batchfldrArr = batchfldr.getFiles();	// get all contained files and folders
		for (var i = 0; i < batchfldrArr.length; i++) {
			if (batchfldrArr[i] instanceof Folder) {
				if (endsWith(batchfldrArr[i].name.toLowerCase(),jobticket.renameAfterDone.toLowerCase()) == false) {
					ok = batchfldrArr[i].rename(batchfldrArr[i].name + jobticket.renameAfterDone);
				}
			}
		}
	}
	if (ok == true) return(0);
	return(-1);
}


function haveMissingFiles(theFile) {
	inxExists = false;
	pdfExists = false;
	jpgExists = false;
	if (overwriteExistingFiles == true) return(true);	// recreate them

	// get the document name without file extension
	var baseDocumentPathName = theFile.fsName;
	if (baseDocumentPathName.indexOf(".indd") > 0) {
		baseDocumentPathName = baseDocumentPathName.substr(0,baseDocumentPathName.lastIndexOf(".indd"));
	}

	// check if needed files exist
	var f = new File(baseDocumentPathName + ".inx");
	if (f.exists == true) inxExists = true;
	f = new File(baseDocumentPathName + ".pdf");
	if (f.exists == true) pdfExists = true;
	f = new File(baseDocumentPathName + ".jpg");
	if (f.exists == true) jpgExists = true;

	if (!inxExists || !pdfExists || !jpgExists) return(true);	// we have missing files - process document
	return(false);
}


function copySettingsFiles() { // copy all settings files from InDesign's scripting folder into the writable communications folder if not already there
	var srcfldr = new Folder(getScriptsPath() + "/" + settingsFilePathAlt);
	var targfldr = new Folder(jobticketFilePath);	// make sure the settings path exists
	var created = targfldr.create();	// make sure the target settings path exists
	var settingsFiles = srcfldr.getFiles("*");

	for (var i = 0; i < settingsFiles.length; i++) {
		if (settingsFiles[i].fsName.indexOf(".DS_Store") >= 0) continue;
		var srcSettingsFilePathName = settingsFiles[i];
		var targSettingsFilePathName = targfldr.fsName + "/" + settingsFiles[i].name;

		var tf = new File(targSettingsFilePathName);	// check if target already exists
		if (tf.exists == true) continue;	// comment this line, if the settings always should be overwritten by application settings

		var sf = new File(srcSettingsFilePathName);
		var success = sf.copy(targSettingsFilePathName);
		/*
		alert("copying settingsFileName: " + settingsFiles[i].name
			+ "\nfrom\nsrcSettingsFilePathName: " + srcSettingsFilePathName
			+ "\nto\ntargSettingsFilePathName: " + targSettingsFilePathName
			+ "\n\nsuccess: " + success
			);
		*/
	}
	return;
}





/****************************
 * check if file is a QuarkXpress file
 */
function isXPressFile(thefile) {
	if (thefile == null) return(false);
	var err = thefile.open("r");
	thefile.encoding = "BINARY"; 
	err = thefile.seek(2,0);				// skip two zero bytes
	var content = "" + thefile.read(15);	// try to read 'MMXPR'
	do {
		// QXP file
		if (content.indexOf("MMXPR") >= 0) return(true);

		// PS file
		if (content.indexOf("PS-") == 0) return(false);
	
		// JEPG file
		if (content.indexOf("JFIF") >= 0) return(false);
	
	} while (false);
	err = thefile.close();
	return(false);
}


/****************************
 * check if file is an InDesign file
 */
function isINDDFile(thefile) {
	if (thefile == null) return(false);
	var err = thefile.open("r");
	thefile.encoding = "BINARY"; 
	err = thefile.seek(16,0);				// skip 16 first bytes
	var content = "" + thefile.read(32);	// try to read 'INDD' signature 'DOCUMENT' starting at byte 16
	err = thefile.close();

	// is INDD file? starting with these bytes
	// 06 06 ED F5 D8 1D 46 E5 BD 31 EF E7 FE 74 B7 1D 44 4F 43 55 4D 45 4E 54

	if (content.indexOf("DOCUMENT") >= 0) return(true);
	return(false);
}


function startsWith (str,subs) { return (str.match("^"+subs)==subs); }

function  endsWith (str,subs) { return (str.match(subs+"$")==subs); }


/****************************
 * floating message window
 */
var messageWindow = null;
function messageClose() {
	if (!is_CS3_or_later) return;
	if (messageWindow != null) {
		messageWindow.close();
		messageWindow = null;
	}
}
function message(mess) {
	if (!is_CS3_or_later) return;
	if ( parseInt(silentMode) < 0 ) return;
	if ((mess == null) || (mess == "")) {
		messageClose();
		return;
	}
	var mymess = mess.replace(/%20/g," ");
	if (messageWindow != null) messageClose();
	messageCreate(mymess);
/*
	if (messageWindow == null) messageCreate(mymess);
	else messageWindow.messtext.text = mymess;
*/
	messageWindow.show();
	return;
}
function messageCreate(mess) {
	messageWindow = new Window('window', 'Progress', undefined, {maximizeButton:false, minimizeButton:false});
	messageWindow.messageGroup = messageWindow.add('group');
	messageWindow.messageGroup.orientation = 'column';
	messageWindow.messageGroup.spacing = 5;
	messageWindow.messageGroup.alignChildren = 'left';
	messageWindow.messageGroup.margins = [20,0,20,0];
	var messarr = mess.split("\n");
	for ( var i = 0; i < messarr.length; i++) messageWindow.messtext = messageWindow.messageGroup.add('statictext', undefined, messarr[i]);
	if (debugMode == 1) {
		var mymess = mess.replace(/\n+/g, " ");
		writeLog(mymess,true);
	}
	return;
}


/****************************
 * write a log message
 */
function writeLog(mess,endline) {
	if (logFilename == "") return;
	if ( (mess == null) || (mess == "") ) return;
	var date = new Date();
	var year = "" + date.getFullYear();
	var month = "" + (date.getMonth() + 1); if (month.length < 2) month = "0" + month;
	var day = "" + date.getDate(); if (day.length < 2) day = "0" + day;
	var hour = "" + date.getHours(); if (hour.length < 2) hour = "0" + hour;
	var minute = "" + date.getMinutes(); if (minute.length < 2) minute = "0" + minute;
	var sec = "" + date.getSeconds(); if (sec.length < 2) sec = "0" + sec;
	var timestamp = "" + year + month + day + " " + hour + ":" + minute + ":" + sec + " " ;
	var logpathname = logFilepath;
	if (logpathname == "") logpathname = "~";
	if ( endsWith(logpathname,"/") == false) logpathname += "/";
	logpathname += logFilename;
	var logFile = new File(logpathname);
	var err = logFile.open("e");
	err = logFile.seek(0,2);	// seek to eof

//	logFile.lineFeed = "windows";
	err = logFile.write(timestamp + mess.replace(/%20/g," "));
	if (endline && !endsWith(mess,"\r") && !endsWith(mess,"\n")) err = logFile.writeln();
	err = logFile.close();
}


/****************************
 * return path to scripting folder
 */
function getScriptsPath() {
	var fullpath = app.activeScript;
	var scriptFile = new File(fullpath);
	return(scriptFile.path);
}


/****************************
 * return path to running script
 */
function getScriptName() {
	var fullpath = app.activeScript;
	var scriptFile = new File(fullpath);
	return(scriptFile.name);
}


/****************************
 * return path portion of full path/name
 */
function getPath(pathname,name) {
	if ((name == null) || (name == "")) return(pathname);
	return(pathname.substr(0,pathname.length-name.length));
}


/****************************
 * return file system name of full path/name
 */
function getFsName(path) {
	if ((path == null) || (path == "")) return("");
	var f = new File(path);
	return(f.fsName);
}


/****************************
 * list object properties
 */
function listProperties(the_prop_to_query, show) {
	var props_arr = the_prop_to_query;
	var props = the_prop_to_query.reflect.properties;
	var pstr = "";
	for (var i = 0; i < props.length; i++) {
		pstr += "Property Name: '" + props[i].name + ",  Type: '" + typeof(props_arr[props[i].name]) + "',  Content: '" + props_arr[props[i].name] + "'\n";

		if (props[i].name == "-------properties") listProperties(props[i], true);
	}
	if ((show == undefined) || (show == true)) alert(pstr);
	return(pstr);
}

/**************************
 * localized message text
 */
var lg=null;
/*
lg[0][0]="english";
lg[0][1]="deutsch";
lg[0][2]="francais";
*/
function setLocale() {
	var num_lg_items=85;	//number of language dependent string array elements
	lg=new Array(num_lg_items);
	for (var i=0; i<num_lg_items; i++) { lg[i]=new Array(3) }	//prepare sub-arrays for 3 languages
	
	lg[0][0]="This export script may not run with this version of InDesign!\nIt is designed to work with InDesign CS5 !\n\nProcessing aborted.";
	lg[0][1]="Dieses Export Script l\u00e4uft nicht mit dieser InDesign Version!\nEs ist ausschliesslich f\u00fcr InDesign CS5 !\n\nVerarbeitung wird abgebrochen.";
	lg[0][2]="Ce script ne fonctionne pas sous cette version de InDesign!\nIl est con\u00e7u pour fonctionner avec InDesign CS5 !\n\nOp\u00e9ration annulu00e9e.";

	lg[1][0]="# Export folder not found at\n\"%%1%%\"\n%%2%%\ncontinuing scanning...";
	lg[1][1]="# Export Ordner nicht gefunden\n\"%%1%%\"\n%%2%%\nScan wird fortgesetzt...";
	lg[1][2]="# Export folder not found at\n\"%%1%%\"\n%%2%%\ncontinuing scanning...";

	lg[2][0]="++ Nothing has been exported from folder\n\"%%1%%\"";
	lg[2][1]="++ Nichts zu exportieren im Ordner\n\"%%1%%\"";
	lg[2][2]="++ Nothing has been exported from folder\n\"%%1%%\"";

	lg[3][0]="++ %%1%% Documents successfully exported from folder \"%%2%%\"";
	lg[3][1]="++ %%1%% Dokumente erfolgreich exportiert aus Ordner \"%%2%%\"";
	lg[3][2]="++ %%1%% Documents successfully exported from folder \"%%2%%\"";

	lg[4][0]="-- %%1%% Documents failed to process from folder \"%%2%%\"";
	lg[4][1]="-- %%1%% Dokumente fehlerhaft bearbeitet aus Ordner \"%%2%%\"";
	lg[4][2]="-- %%1%% Documents failed to process exported from folder \"%%2%%\"";

	lg[5][0]="Sleeping %%1%%ms...\nWaiting for next JobTicket.\nPress %%2%% to stop processing.";
	lg[5][1]="Schlafe %%1%%ms...\nWarte auf folgendes JobTicket.\nDie Tasten %%2%% brechen die Verarbeitung ab.";
	lg[5][2]="Sleeping %%1%%ms...\nWaiting for next JobTicket.\nPress %%2%% to stop processing.";

	lg[6][0]="Sleeping %%1%%ms...\nWaiting for next loop.\nPress %%2%% to stop processing.";
	lg[6][1]="Schlafe %%1%%ms...\nWarte auf folgenden Durchgang.\nDie Tasten %%2%% brechen die Verarbeitung ab.";
	lg[6][2]="Sleeping %%1%%ms...\nWaiting for next loop.\nPress %%2%% to stop processing.";

	lg[7][0]="JobTicket file \"%%1%%\" not found at\n\"%%2%%\".\nProcessing stopped.";
	lg[7][1]="JobTicket File \"%%1%%\" nicht gefunden in\n\"%%2%%\".\nVerarbeitung wird abgebrochen.";
	lg[7][2]="JobTicket file \"%%1%%\" not found at\n\"%%2%%\".\nProcessing stopped.";

	lg[8][0]="No active JobTickets in file \"%%1%%\"\nJobTicket path \"%%2%%\"\nProcessing stopped.";
	lg[8][1]="Keine activen JobTickets in Datei \"%%1%%\"\nJobTicket Pfad \"%%2%%\"\nVerarbeitung wird abgebrochen.";
	lg[8][2]="No active JobTickets in file \"%%1%%\"\nJobTicket path \"%%2%%\"\nProcessing stopped.";

	lg[20][0]="Calling Export script: File \"%%1%%\" at folder \"%%2%%\" - Settings \"%%3%%\"";
	lg[20][1]="Aufruf Export Script: Datei \"%%1%%\" in Ordner \"%%2%%\" - Settings \"%%3%%\"";
	lg[20][2]="Calling Export script: File \"%%1%%\" at folder \"%%2%%\" - Settings \"%%3%%\"";

	lg[21][0]="Export finished with Code %%1%%\n";
	lg[21][1]="Export beendet mit Code %%1%%\n";
	lg[21][2]="Export finished with Code %%1%%\n";

	lg[22][0]="Document \"%%1%%\" closed.\n";
	lg[22][1]="Dokument \"%%1%%\" geschlossen.\n";
	lg[22][2]="Document \"%%1%%\" closed.\n";

	lg[23][0]="# Error %%1%% exporting file \"%%2%%\" at folder \"%%3%%\".";
	lg[23][1]="# Fehler %%1%% beim Export der Datei \"%%2%%\" aus Ordner \"%%3%%\".";
	lg[23][2]="# Error %%1%% exporting file \"%%2%%\" at folder \"%%3%%\".";

	lg[24][0]="BatchXSLT Transformer successfully started.\n";
	lg[24][1]="BatchXSLT Transformer erfolgreich gestartet.\n";
	lg[24][2]="BatchXSLT Transformer successfully started.\n";

	lg[25][0]="# Error starting BatchXSLT Transformer.";
	lg[25][1]="# Fehler beim Starten des BatchXSLT Transformer.";
	lg[25][2]="# Error starting BatchXSLT Transformer.";

	lg[26][0]="# Error opening document \"%%1%%\" at folder \"%%2%%\".";
	lg[26][1]="# Fehler beim Export der Datei \"%%1%%\" aus Ordner \"%%2%%\".";
	lg[26][2]="# Error opening document \"%%1%%\" at folder \"%%2%%\".";

	lg[27][0]="# Error reading settings file \"%%1%%\"  defined in JobTicket \"%%2%%\".";
	lg[27][1]="# Fehler Lesen der Settings Datei \"%%1%%\"  definiert in JobTicket \"%%2%%\".";
	lg[27][2]="# Error reading settings file \"%%1%%\"  defined in JobTicket \"%%2%%\".";

}

function loc (which,par1,par2,par3,par4) {
	if (which >= lg.length) return("");
	if (lg[which][appLanguage].indexOf("%%") < 0) 	return(lg[which][appLanguage]);
	var tx = lg[which][appLanguage];
	if (par1 != null) tx = tx.replace(/%%1%%/gi,par1);
	if (par2 != null) tx = tx.replace(/%%2%%/gi,par2);
	if (par3 != null) tx = tx.replace(/%%3%%/gi,par3);
	if (par4 != null) tx = tx.replace(/%%4%%/gi,par4);
	return(tx);
}

